#!/bin/sh
# Installs Yocto to NAND/eMMC
# set -e


MEDIA=/opt/images/Yocto
SPL_IMAGE=SPL-nand
UBOOT_IMAGE=u-boot.img-nand
KERNEL_IMAGE=uImage
KERNEL_DTB=""
ROOTFS_DEV=""

# $1 is the full path of the config file
set_fw_env_config_to_nand()
{
	sed -i "/mmcblk/ s/^#*/#/" $1
	sed -i "s/#*\/dev\/mtd/\/dev\/mtd/" $1

	MTD_DEV=`grep /dev/mtd $1 | cut -f1 | sed "s/\/dev\/*//"`
	MTD_ERASESIZE=$(printf 0x%x $(cat /sys/class/mtd/${MTD_DEV}/erasesize))
	awk -i inplace -v n=4 -v ERASESIZE="${MTD_ERASESIZE}" '/\/dev\/mtd/{$(n)=ERASESIZE}1' $1
}

set_fw_utils_to_nand_on_sd_card()
{
	# Adjust u-boot-fw-utils for NAND flash on the SD card
	if [ `readlink /sbin/fw_printenv` != "/sbin/fw_printenv-nand" ]; then
		ln -sf /sbin/fw_printenv-nand /sbin/fw_printenv
	fi

	if [ -f /etc/fw_env.config ]; then
		set_fw_env_config_to_nand /etc/fw_env.config
	fi
}

install_bootloader()
{
	echo
	echo "Installing booloader"

	if [ ! -f $MEDIA/$UBOOT_IMAGE ] ; then
		echo "ERROR: \"$MEDIA/$UBOOT_IMAGE\" does not exist"
		exit 1
	fi

	if [ ! -f $MEDIA/$SPL_IMAGE ] ; then
		echo "ERROR: \"$MEDIA/$SPL_IMAGE\" does not exist"
		exit 1
	fi

	flash_erase /dev/mtd0 0 0 2> /dev/null
	kobs-ng init -x $MEDIA/$SPL_IMAGE --search_exponent=1 -v > /dev/null

	flash_erase /dev/mtd1 0 0 2> /dev/null
	nandwrite -p /dev/mtd1 $MEDIA/$UBOOT_IMAGE

	if [ $ROOTFS_DEV = "emmc" ] ; then
		echo
		echo "Setting U-Boot enviroment variables"
		set_fw_utils_to_nand_on_sd_card
		fw_setenv rootfs_device emmc  2> /dev/null

		if [ $swupdate = 1 ] ; then
			fw_setenv mmcrootpart 1
			fw_setenv boot_device emmc
			fw_setenv bootdir /boot
		fi
	fi
}

install_kernel()
{
	if [ ! -f $MEDIA/$KERNEL_IMAGE ] ; then
		echo "ERROR: \"$MEDIA/$KERNEL_IMAGE\" does not exist"
		exit 1
	fi
	echo
	echo "Installing kernel"
	flash_erase /dev/mtd2 0 0 2> /dev/null
	nandwrite -p /dev/mtd2 $MEDIA/$KERNEL_IMAGE > /dev/null
	nandwrite -p /dev/mtd1 -s 0x1e0000 $MEDIA/$KERNEL_DTB > /dev/null
}

install_rootfs_to_nand()
{
	if [ ! -f $MEDIA/$ROOTFS_IMAGE ] ; then
		echo "ERROR: \"$MEDIA/$ROOTFS_IMAGE\" does not exist"
		exit 1
	fi
	echo
	echo "Installing UBI rootfs"
	ubiformat /dev/mtd3 -f $MEDIA/$ROOTFS_IMAGE -y
}

install_rootfs()
{
	if [ $ROOTFS_DEV != "emmc" ] ; then
		install_rootfs_to_nand
	else
		/usr/bin/install_yocto_emmc.sh ${EMMC_EXTRA_ARGS}
	fi
}

usage()
{
	echo
	echo "This script installs Yocto on the SOM's internal storage device/s"
	echo
	echo " Usage: $0 OPTIONS"
	echo
	echo " OPTIONS:"
	echo " -b <mx6cb|scb|dart>	carrier Board model (MX6CustomBoard/SOLOCustomBoard/DART-MX6) - mandartory parameter."
	echo " -t <cap|res>		Touchscreen model (capacitive/resistive) - mandatory in case of MX6CustomBoard; ignored otherwise."
	echo " -r <nand|emmc>		Rootfs device (NAND/eMMC) - mandatory in case of MX6CustomBoard/SOLOCustomBoard; ignored in case of DART-MX6."
	echo " -u			create two rootfs partitions (for swUpdate double-copy) - ignored in case of NAND rootfs device."
	echo
}

finish()
{
	sync
	echo
	echo "Yocto installed successfully"
	exit 0
}


echo "*** LC MX6 Yocto eMMC/NAND Recovery ***"
echo

### Do not process options
###while getopts :b:t:r:u OPTION;
###do
###	case $OPTION in
###	b)
###		BOARD=$OPTARG
###		;;
###	t)
###		TOUCHSCREEN=$OPTARG
###		;;
###	r)
###		ROOTFS_DEV=$OPTARG
###		;;
###	u)
###		swupdate=1
###		EMMC_EXTRA_ARGS="-u"
###		;;
###	*)
###		usage
###		exit 1
###		;;
###	esac
###done

STR=""
### Force options
BOARD="mx6cb"
TOUCHSCREEN="res"
KERNEL_DTB=imx6dl-var-som-solo-res.dtb
ROOTFS_DEV="nand"
ROOTFS_IMAGE=rootfs_128kbpeb.ubi

if [ $BOARD = "mx6cb" ] ; then
	STR="MX6CustomBoard"
elif [ $BOARD = "scb" ] ; then
	STR="SOLOCustomBoard"
elif [ $BOARD = "dart" ] ; then
	STR="DART-MX6"
else
	usage
	exit 1
fi

printf "Carrier board: "
echo $STR

if [ $BOARD = "mx6cb" ] ; then
	if [ $TOUCHSCREEN = "cap" ] ; then
		STR="Capacitive"
	elif [ $TOUCHSCREEN = "res" ] ; then
		STR="Resistive"
	else
		usage
		exit 1
	fi
	printf "Touchscreen model: "
	echo $STR
fi

STR="NAND"
printf "Installing rootfs to: "
echo $STR


###printf "Installing Device Tree file: "
###echo $KERNEL_DTB

###install_bootloader
###	install_kernel

install_rootfs

finish
